/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpFixedKey.c,v 1.3 2000/11/29 22:14:57 hal Exp $
____________________________________________________________________________*/
/*
 * pgpFixedKey.c -- A single, fixed PGP key for decryption operations.
 * Also contains fixed primes and generators for discrete log keys which
 * can share these.
 */
#include "pgpConfig.h"

#include "pgpUsuals.h"
#include "pgpFixedKey.h"

/* Fixed p, q, g values for DSA keys */

#if 0
static PGPByte const DSA1024p[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4,
	0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D,
	0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D,
	0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3,
	0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C,
	0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1,
	0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA,
	0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x50, 0x9B,
	0x02, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0xB0, 0xB6, 0x51,
	0xE2, 0xBA, 0xE5, 0x4E, 0x44, 0x39, 0x7C, 0x9F, 0x2A, 0x5E,
	0x31, 0x2F, 0xF6, 0xDB, 0x5C, 0x75, 0xAE, 0x81

};
static PGPByte const DSA1024q[] = {
	0xFF, 0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26,
	0xBA, 0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0xB1
};
#endif
static PGPByte const DSA1536p[] = {
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F, 0x3D, 0x3D,
	0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59, 0xCD, 0xFD,
	0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5,
	0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3, 0x5B, 0x9E,
	0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0, 0xBF, 0x46,
	0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49,
	0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3,
	0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0,
	0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C,
	0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A,
	0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C,
	0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8,
	0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x47, 0x2C, 0xCB, 0xA6, 0x82,
	0xA5, 0xEB, 0xA2, 0x84, 0x3E, 0xE3, 0x3D, 0x56, 0xE7, 0xA9,
	0xF3, 0xE1, 0xC4, 0x3D, 0x22, 0x98, 0x80, 0xF3, 0xC3, 0x08,
	0x49, 0x8F
};
static PGPByte const DSA1536q[] = {
	0x3F, 0xFD, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C, 0xBD,
	0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB, 0xA2, 0x5E, 0xC3,
	0x55, 0xE9, 0x27, 0xC2, 0x43
};
static PGPByte const DSA2048p[] = {
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2,
	0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11,
	0x39, 0x4F, 0xB6, 0xF1, 0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA,
	0x01, 0xA7, 0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C,
	0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17,
	0xD4, 0xA1, 0xD3, 0x50, 0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76,
	0x52, 0x4F, 0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC,
	0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9,
	0x66, 0xFF, 0x15, 0xF9, 0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01,
	0xAA, 0xD3, 0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88,
	0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4,
	0x91, 0x07, 0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D,
	0x0F, 0x7C, 0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D,
	0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD,
	0x7C, 0x17, 0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33,
	0x21, 0x2C, 0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86,
	0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C,
	0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x51, 0xB8, 0x4F, 0xCF,
	0x7E, 0xC9, 0x6A, 0x96, 0xFA, 0x9E, 0x4E, 0xDC, 0x9F, 0x93,
	0x37, 0x21, 0x13, 0x1D, 0xE3, 0xDD, 0x3D, 0xE0, 0x7D, 0x1D,
	0x6B, 0xCE, 0x09, 0x83, 0x11, 0xB5
};
static PGPByte const DSA2048q[] = {
	0x01, 0xFE, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0x5D, 0x41,
	0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA,
	0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0xCF
};
static PGPByte const DSA3072p[] = {
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2, 0xC4, 0xF0,
	0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E, 0x92, 0x13, 0x61, 0xF6,
	0xDB, 0xEB, 0x25, 0x0E, 0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8,
	0x7C, 0x79, 0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60, 0xC3, 0xE3,
	0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28, 0x26, 0x0B, 0xAE, 0xA9,
	0xAE, 0xB3, 0x65, 0x0F, 0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C,
	0xD6, 0xAB, 0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4, 0x40, 0xE9,
	0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D, 0x02, 0x36, 0xC1, 0x26,
	0xCB, 0x68, 0x5E, 0x9D, 0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E,
	0x2D, 0xED, 0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32, 0xF6, 0x42,
	0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2, 0xBA, 0xD6,
	0xA9, 0x42, 0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F,
	0xB6, 0xF1, 0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3, 0xD0, 0x07,
	0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C, 0x64, 0x86,
	0x49, 0xF8, 0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1,
	0xD3, 0x50, 0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59,
	0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC,
	0x83, 0xC5, 0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF,
	0x15, 0xF9, 0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0,
	0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58,
	0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07,
	0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9,
	0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB,
	0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17,
	0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11,
	0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03,
	0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C, 0xDB, 0x13,
	0xD8, 0xA2, 0x1A, 0x8F, 0x32, 0x6B, 0x11, 0xEB, 0xA3, 0xF2,
	0x71, 0x5E, 0xF5, 0xC5, 0x43, 0x6D, 0x84, 0x4B, 0xF6, 0xFE,
	0x52, 0xB9, 0x60, 0x5A, 0x8E, 0x09, 0xCE, 0xCB, 0xFE, 0x94,
	0xBC, 0xEF, 0x23, 0x5B
};
static PGPByte const DSA3072q[] = {
	0x3F, 0xEB, 0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C, 0xBD, 0x89,
	0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB, 0xA2, 0x5E, 0xC3, 0x55,
	0xE9, 0x27, 0xC3, 0x1D

};
static PGPByte const DSA4096p[] = {
	0xF9, 0x18, 0xA0, 0x7E, 0x5A, 0x06, 0x61, 0x7A, 0x43, 0x90,
	0x95, 0xDC, 0x05, 0x6C, 0x87, 0x86, 0xEC, 0x61, 0xEC, 0xCD,
	0x45, 0x1F, 0x0E, 0xD8, 0xE0, 0xA3, 0x79, 0xC6, 0xC9, 0xDC,
	0x7A, 0x0B, 0xAC, 0xE4, 0x3F, 0xE3, 0x46, 0x94, 0xB6, 0x30,
	0x4A, 0x53, 0xD7, 0x7C, 0x02, 0x16, 0x48, 0x80, 0xB5, 0x15,
	0xE5, 0x29, 0x99, 0xA9, 0x9F, 0x07, 0x74, 0xD3, 0xFF, 0xE3,
	0xA1, 0xC5, 0x96, 0x20, 0x4E, 0x98, 0x65, 0xB8, 0xD8, 0x0D,
	0xEE, 0x10, 0x5D, 0xAB, 0xB6, 0x17, 0x1C, 0x51, 0xD8, 0x50,
	0xCA, 0x22, 0x57, 0x43, 0x29, 0xBE, 0x95, 0xE8, 0x56, 0x2B,
	0x38, 0x78, 0x5C, 0x0B, 0xDB, 0xF8, 0x4C, 0x4D, 0xD5, 0xE3,
	0xAA, 0x46, 0xCC, 0xFB, 0xCE, 0x17, 0xE8, 0x2A, 0x9D, 0x14,
	0x61, 0xE3, 0x84, 0xA9, 0x4F, 0xD1, 0x83, 0x84, 0xA8, 0x79,
	0xB6, 0xEF, 0x8F, 0xA7, 0x43, 0x46, 0x08, 0xC6, 0xCC, 0x1D,
	0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2, 0xC4, 0xF0, 0xE8, 0x8E,
	0x13, 0x67, 0x97, 0x4E, 0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB,
	0x25, 0x0E, 0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1, 0x73, 0xC4,
	0x83, 0x11, 0x26, 0x2B, 0x43, 0x60, 0xC3, 0xE3, 0xE8, 0xD6,
	0x0A, 0xFD, 0xA1, 0x28, 0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3,
	0x65, 0x0F, 0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F, 0x74, 0x84,
	0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4, 0x40, 0xE9, 0xD7, 0xD2,
	0x77, 0xB6, 0x42, 0x2D, 0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68,
	0x5E, 0x9D, 0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B, 0x18, 0x8E,
	0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32, 0xF6, 0x42, 0x57, 0xB7,
	0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42,
	0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7, 0x56, 0xA3,
	0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3, 0xD0, 0x07, 0xC6, 0xCB,
	0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8,
	0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F, 0x3D, 0x3D,
	0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59, 0xCD, 0xFD,
	0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5,
	0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3, 0x5B, 0x9E,
	0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0, 0xBF, 0x46,
	0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49,
	0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3,
	0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0,
	0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C,
	0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A,
	0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C,
	0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8,
	0x07, 0x29, 0x88, 0x50, 0x05, 0x4B, 0xFE, 0x24, 0x36, 0x96,
	0x47, 0xAB, 0x40, 0x4A, 0xA2, 0xE5, 0xB3, 0xCB, 0x0E, 0x0F,
	0x2A, 0x1C, 0x45, 0x17, 0x3D, 0xD9, 0x5D, 0xED, 0x9F, 0x0D,
	0x33, 0xDA, 0x04, 0xC9, 0x19, 0xA9, 0x9D, 0x3F, 0x40, 0xF3,
	0xF4, 0x2D
};
static PGPByte const DSA4096q[] = {
	0x01, 0xFF, 0x0C, 0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x50,
	0x9B, 0x02, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0x5D, 0x41,
	0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA,
	0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0x65

};

static struct {
	unsigned	bits;
	PGPByte const *p;
	unsigned	psize;
	PGPByte const *q;
	unsigned	qsize;
} DSAtab[] = {
#if 0
	{1024, DSA1024p, sizeof(DSA1024p),
	 	DSA1024q, sizeof(DSA1024q)},
#endif
	{1536, DSA1536p, sizeof(DSA1536p),
	 	DSA1536q, sizeof(DSA1536q)},
	{2048, DSA2048p, sizeof(DSA2048p),
	 	DSA2048q, sizeof(DSA2048q)},
	{3072, DSA3072p, sizeof(DSA3072p),
	 	DSA3072q, sizeof(DSA3072q)},
	{4096, DSA4096p, sizeof(DSA4096p),
	 	DSA4096q, sizeof(DSA4096q)}
};


/* Return >0 if we find one, and fill in pointers, else return 0 */
int
pgpDSAfixed (unsigned bits, PGPByte const **p, PGPSize *plen,
	PGPByte const **q, PGPSize *qlen)
{
	unsigned i;

	for (i=0; i<sizeof(DSAtab)/sizeof(DSAtab[0]); ++i) {
		if (DSAtab[i].bits > bits)
			return 0;
		if (DSAtab[i].bits == bits) {
			if (p)
				*p = DSAtab[i].p;
			if (plen)
				*plen = DSAtab[i].psize;
			if (q)
				*q = DSAtab[i].q;
			if (qlen)
				*qlen = DSAtab[i].qsize;
			return 1;
		}
	}
	return 0;
}



/*
 * Fixed p, g values for ElG keys.  The p values are all "strong" primes,
 * such that (p-1)/2 is also prime.  This makes 2 an appropriate generator
 * in each case.
 */

#if 0
static PGPByte const ElG1024p[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2F, 0x78, 0xC7
};
static PGPByte const ElG1024g[] = {
	0x02
};
#endif

static PGPByte const ElG1536p[] = {
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x58, 0x7C, 0x47
};
static PGPByte const ElG1536g[] = {
	0x02
};
static PGPByte const ElG2048p[] = {
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x32, 0x0B, 0x3B
};
static PGPByte const ElG2048g[] = {
	0x02
};
static PGPByte const ElG3072p[] = {
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x4C, 0x32, 0x6F
};
static PGPByte const ElG3072g[] = {
	0x02
};
static PGPByte const ElG4096p[] = {
	0xF9, 0x18, 0xA0, 0x7E, 0x5A, 0x06, 0x61, 0x7A,
	0x43, 0x90, 0x95, 0xDC, 0x05, 0x6C, 0x87, 0x86,
	0xEC, 0x61, 0xEC, 0xCD, 0x45, 0x1F, 0x0E, 0xD8,
	0xE0, 0xA3, 0x79, 0xC6, 0xC9, 0xDC, 0x7A, 0x0B,
	0xAC, 0xE4, 0x3F, 0xE3, 0x46, 0x94, 0xB6, 0x30,
	0x4A, 0x53, 0xD7, 0x7C, 0x02, 0x16, 0x48, 0x80,
	0xB5, 0x15, 0xE5, 0x29, 0x99, 0xA9, 0x9F, 0x07,
	0x74, 0xD3, 0xFF, 0xE3, 0xA1, 0xC5, 0x96, 0x20,
	0x4E, 0x98, 0x65, 0xB8, 0xD8, 0x0D, 0xEE, 0x10,
	0x5D, 0xAB, 0xB6, 0x17, 0x1C, 0x51, 0xD8, 0x50,
	0xCA, 0x22, 0x57, 0x43, 0x29, 0xBE, 0x95, 0xE8,
	0x56, 0x2B, 0x38, 0x78, 0x5C, 0x0B, 0xDB, 0xF8,
	0x4C, 0x4D, 0xD5, 0xE3, 0xAA, 0x46, 0xCC, 0xFB,
	0xCE, 0x17, 0xE8, 0x2A, 0x9D, 0x14, 0x61, 0xE3,
	0x84, 0xA9, 0x4F, 0xD1, 0x83, 0x84, 0xA8, 0x79,
	0xB6, 0xEF, 0x8F, 0xA7, 0x43, 0x46, 0x08, 0xC6,
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xEB, 0x3D, 0xD6, 0x17
};
static PGPByte const ElG4096g[] = {
	0x02
};

static struct {
	unsigned	bits;
	PGPByte const *p;
	unsigned	psize;
	PGPByte const *g;
	unsigned	gsize;
} ElGtab[] = {
#if 0
	{1024, ElG1024p, sizeof(ElG1024p),
		ElG1024g, sizeof(ElG1024g)},
#endif
	{1536, ElG1536p, sizeof(ElG1536p),
		ElG1536g, sizeof(ElG1536g)},
	{2048, ElG2048p, sizeof(ElG2048p),
		ElG2048g, sizeof(ElG2048g)},
	{3072, ElG3072p, sizeof(ElG3072p),
		ElG3072g, sizeof(ElG3072g)},
	{4096, ElG4096p, sizeof(ElG4096p),
		ElG4096g, sizeof(ElG4096g)}
};


/* Return >0 if we find one, and fill in pointers, else return 0 */
int
pgpElGfixed (unsigned bits, PGPByte const **p, PGPSize *plen,
	PGPByte const **g, PGPSize *glen)
{
	unsigned i;

	for (i=0; i<sizeof(ElGtab)/sizeof(ElGtab[0]); ++i) {
		if (ElGtab[i].bits > bits)
			return 0;
		if (ElGtab[i].bits == bits) {
			if (p)
				*p = ElGtab[i].p;
			if (plen)
				*plen = ElGtab[i].psize;
			if (g)
				*g = ElGtab[i].g;
			if (glen)
				*glen = ElGtab[i].gsize;
			return 1;
		}
	}
	return 0;
}

